/* $Id: meas.c,v 1.101 1999/10/04 23:42:25 donwm Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

/* This file groups all the functions associated with the control of
   the measurement loop.  The vxi registers accessed by the functions
   are CONTrol, STATus. */

#include "sema.h"

#define DEFAULT_WAIT	5.0	/* wait_all_state */
#define	MIN_WAIT	3.0	/* init_measure_finish, arm_measure */
#define MAX_SETTLING	240	/* max settling time for sca, actual max is
				   is 70, but this is max possible points
				   that accumulate in internal RAM buffer
				   before DMA transfer*/

/*
 *********************************************************************
 Get measurement state of module.  Returns 0 if all ok, else return
 negative error number.
 *********************************************************************
 */
static SHORTSIZ16
i1432_read_state(E1432_MODULE_LIST_NODE * mn, SHORTSIZ16 * state)
{
    SHORTSIZ16 error;
    LONGSIZ32 ltmp;

    error = i1432_direct_read32_register(mn, E1432_MEAS_STATE_REG,
					 &ltmp);
    if (error)
	return error;
    *state = (SHORTSIZ16) (ltmp & E1432_MEAS_STATE_MASK);
    return 0;
}

static SHORTSIZ16
wait_cleanup(E1432ID hw, E1432_MODULE_LIST_NODE *mn,
	     SHORTSIZ16 state, SHORTSIZ16 error)
{
    SHORTSIZ16 bits, ter;

    if (error)
    {
	ter = i1432_error_check(hw, i1432_get_chan_from_module(mn));
	if (ter)
	    return ter;

	if (i1432_print_errors)
	    PRINTF(("wait_state(0x%x) timeout error:\n", state));
	ter = i1432_direct_read_register(mn, E1432_STATUS_REG, &bits);
	if (ter)
	{
	    if (i1432_print_errors)
		PRINTF(("status read after timeout failed!?\n"));
	    return ter;
	}
	if (i1432_print_errors)
	    PRINTF(("module at %d, status = 0x%x\n",
		    mn->la, bits & 0xffff));
	return error;
    }
    return 0;
}

/*
 *********************************************************************
 Wait for one module to reach the specified state.
 *********************************************************************
 */
static SHORTSIZ16
wait_all_state_mod(E1432ID hw, E1432_MODULE_LIST_NODE *mn,
		   SHORTSIZ16 state, FLOATSIZ64 timeout)
{
    SHORTSIZ16 error;
    char    err[40];

    /* Wait for state to match parameter */
    (void) sprintf(err, "module at %d never reached state 0x%x\n",
		   mn->la, state);
    error = i1432_error_wait32_bits_match(mn, E1432_MEAS_STATE_REG,
					  E1432_MEAS_STATE_MASK,
					  state, timeout, err);
    return wait_cleanup(hw, mn, state, error);
}

/*
 *********************************************************************
 Wait for one module to reach a state other than the specified state.
 *********************************************************************
 */
static SHORTSIZ16
wait_all_not_state_mod(E1432ID hw, E1432_MODULE_LIST_NODE *mn,
		       SHORTSIZ16 state, FLOATSIZ64 timeout)
{
    SHORTSIZ16 error;
    char    err[40];

    /* Wait for state to not match parameter */
    (void) sprintf(err, "module at %d never left state 0x%x\n",
		   mn->la, state);
    error = i1432_error_wait32_bits_no_match(mn, E1432_MEAS_STATE_REG,
					     E1432_MEAS_STATE_MASK,
					     state, timeout, err);
    return wait_cleanup(hw, mn, state, error);
}

/*
 *********************************************************************
 Wait for one module to reach a state greater than the specified
 state.
 *********************************************************************
 */
static SHORTSIZ16
wait_all_greater_state_mod(E1432ID hw, E1432_MODULE_LIST_NODE *mn,
			   SHORTSIZ16 state, FLOATSIZ64 timeout)
{
    SHORTSIZ16 error, bits;
    FLOATSIZ64 startTime, currentTime;
    char    err[40];

    i1432_get_time(&startTime);
    do
    {
	i1432_get_time(&currentTime);

	error = i1432_direct_read_register(mn, E1432_STATUS_REG,
					   &bits);
	if (error)
	    return error;
	if ((bits & E1432_STATUS_ERR_N) == 0)
	    return wait_cleanup(hw, mn, state, -1);

	error = i1432_read_state(mn, &bits);
	if (error)
	    return error;
	if (bits > state)
	    return 0;
    }
    while (currentTime - startTime < timeout);

    (void) sprintf(err, "waiting for mod at %d leave state 0x%x",
		   mn->la, state);
    i1432_error_info = err;
    error = wait_cleanup(hw, mn, state, ERR1432_STATUS_READ_TIMEOUT);
    return i1432_la_print_error(mn->la, error);
}

/*
 *********************************************************************
 Return whan all of a group gets to the specified state.  Returns 0 if
 all ok, else return negative error number.
 *********************************************************************
 */
static SHORTSIZ16
i1432_wait_all_state(E1432ID hw, SHORTSIZ16 groupID, SHORTSIZ16 state,
		     FLOATSIZ64 timeout)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod;

    if (groupID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, groupID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    mn = gn->modlist[mod];
	    error = wait_all_state_mod(hw, mn, state, timeout);
	    if (error)
		return error;
	}
    }
    else
    {
	/* Do single module */
	error = i1432_get_module_from_chan(hw, groupID, &mn);
	if (error)
	    return error;
	error = wait_all_state_mod(hw, mn, state, timeout);
	if (error)
	    return error;
    }
    return 0;
}

/*
 *********************************************************************
 Return whan all board of a group are not at state.  Returns 0 if all
 ok, else return negative error number.
 *********************************************************************
 */
static SHORTSIZ16
i1432_wait_all_not_state(E1432ID hw, SHORTSIZ16 groupID,
			 SHORTSIZ16 state, FLOATSIZ64 timeout)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod;

    if (groupID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, groupID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    mn = gn->modlist[mod];
	    error = wait_all_not_state_mod(hw, mn, state, timeout);
	    if (error)
		return error;
	}
    }
    else
    {
	/* Do single module */
	error = i1432_get_module_from_chan(hw, groupID, &mn);
	if (error)
	    return error;
	error = wait_all_not_state_mod(hw, mn, state, timeout);
	if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 Return whan all board of a group get to greater than state.  Returns
 0 if all ok, else return negative error number.
 *********************************************************************
 */
static SHORTSIZ16
i1432_wait_all_greater_state(E1432ID hw, SHORTSIZ16 groupID,
			     SHORTSIZ16 state, FLOATSIZ64 timeout)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod;

    if (groupID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, groupID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    mn = gn->modlist[mod];
	    error = wait_all_greater_state_mod(hw, mn, state,
					       timeout);
	    if (error)
		return error;
	}
    }
    else
    {
	/* Do single module */
	error = i1432_get_module_from_chan(hw, groupID, &mn);
	if (error)
	    return error;
	error = wait_all_greater_state_mod(hw, mn, state, timeout);
	if (error)
	    return error;
    }

    return 0;
}

static SHORTSIZ16
map_one_fifo(E1432ID hw, E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 data_port, error;

    error = e1432_get_data_port(hw, mn->chan_id, &data_port);
    if (error)
	return error;

    /* No need for FIFO if data is just going to local bus */
    if (data_port == E1432_SEND_PORT_LBUS)
	/*(void) printf("Skipping A24 FIFO map at la %d\n", mn->la);*/
	return 0;

    error = i1432_imap(mn, 1);
    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_unmap_one_fifo(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    error = i1432_iunmap(mn, 1);
    if (error)
	return error;

    return 0;
}

/*
 *********************************************************************
 The code in map_one_fifo assumes that e1432_cached_parm_update was
 already called, and that this cleared the module's d32 flag if the
 module is using local bus.
 ********************************************************************* */
static SHORTSIZ16
map_fifos(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    int     mod;
    SHORTSIZ16 error;

    if (ID < 0)
    {
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
#ifdef	HAVE_SICL
	    error = i1432_sicl_imap_setup2(gn->modlist[mod]);
	    if (error)
		return error;
#endif
	    error = map_one_fifo(hw, gn->modlist[mod]);
	    if (error)
		return error;
	}
    }
    else
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
#ifdef	HAVE_SICL
	error = i1432_sicl_imap_setup2(mn);
	if (error)
	    return error;
#endif
	error = map_one_fifo(hw, mn);
	if (error)
	    return error;
    }

    return 0;
}

static SHORTSIZ16
unmap_fifos(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    int     mod;
    SHORTSIZ16 error;

    if (ID < 0)
    {
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    error = i1432_unmap_one_fifo(gn->modlist[mod]);
	    if (error)
		return error;
	}
    }
    else
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
	error = i1432_unmap_one_fifo(mn);
	if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 e1432_reset_measure used to leave the module's state machine
 unfrozen, so a module could immediately start a measurement.  We
 changed that so it leaves the measurement state machine frozen, to
 avoid multi-module problems of one module unintentionally moving
 another module's state machine.  The measurement state machine must
 be unfrozen before a measurement can start, using this function.
 *********************************************************************
 */
static SHORTSIZ16
unreset_measure(E1432ID hw, SHORTSIZ16 ID)
{
    return i1432_set_mod(hw, ID, -1, E1432_CMD_SET_FREEZE, 0);
}

/*
 *********************************************************************
 Reset the measurement state of the group or channel.  In the Octane
 library, this only allows groups.  I see no reason to make this
 restriction, so I'm not doing it.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_reset_measure(E1432ID hw, SHORTSIZ16 ID)
{
#ifndef	NO_D32
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    int     mod;
#endif
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_reset_measure(0x%p, %d)\n", hw, ID));
    /* Note dummy parameter at end, which is not needed for the RESET
       MEASURE commands */
    error = i1432_set_mod_nowait(hw, ID, -1, E1432_CMD_RESET_MEASURE_START, 0);
    if (error)
	return error;
    error = i1432_set_mod(hw, ID, -1, E1432_CMD_RESET_MEASURE_MIDDLE, 0);
    if (error)
	return error;
    error = i1432_set_mod(hw, ID, -1, E1432_CMD_RESET_MEASURE_FINISH, 0);
    if (error)
	return error;

#ifndef	NO_D32
    /* Restore d32 setting, which may have been set to zero during a
       measurement if we were using local bus. */
#ifdef HAVE_VTL
    if (!i1432_visa_d16) /* don't set d32 if VISA is 16 bit only  */
    {
#endif
        if (ID < 0)
        {
	    gn = i1432_get_group_node(hw, ID);
	    for (mod = 0; mod < gn->modcount; mod++)
	        gn->modlist[mod]->d32 = 1;
        }
        else
        {
	    error = i1432_get_module_from_chan(hw, ID, &mn);
	    if (error)
	        return error;
	    mn->d32 = 1;
        }
#ifdef HAVE_VTL
    }
#endif
#endif

    /* Clear request spectrum flags */
    error = i1432_reset_spectrum(hw, ID);
    if (error)
	return error;

    error = unmap_fifos(hw, ID);
    if (error)
	return error;
    return 0;
}

/*
 *********************************************************************
 This is called by e1432_init_measure_to_booted, after syncing the
 module and waiting for the state to change.  This catches setup
 errors that can't be detected at setup time.
 *********************************************************************
 */
static SHORTSIZ16
i1432_error_check_group(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod;

    TRACE_PRINTF(1, ("i1432_error_check_group(0x%p, %d)\n", hw, ID));

    /* If single channel, just do that */
    if (ID >= 0)
	return i1432_error_check(hw, ID);

    /* Iterate through group */
    gn = i1432_get_group_node(hw, ID);
    for (mod = 0; mod < gn->modcount; mod++)
    {
	/* Do one module */
	mn = gn->modlist[mod];
	error = i1432_error_check(hw,
				  i1432_get_chan_from_module(mn));
	if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 Pull sync on system module of group ID.  Returns 0 if all ok, else
 return negative error number.
 *********************************************************************
 */
SHORTSIZ16
i1432_sync_system(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    /* get system module */
    error = i1432_get_system_module(hw, ID, &mn);
    if (error)
	return error;
    if (mn != NULL)
    {
	error = i1432_write_cmd0(hw, i1432_get_chan_from_module(mn),
				 E1432_CMD_SYNC);
	if (error)
	    return error;
    }
    return 0;
}

/*
 *********************************************************************
 This code roughly matches what the E143x firmware does in
 subst/dsp.c.  It's messy, partly because the definition of how
 calc_data works is unnecessarily complicated.  If we were to switch
 to using meas_type to indicate what measurement is running, this
 would be much simpler.

 This must be called after the cached channel parameters have been set
 up, so that we can use the calc_data and active fields of the channel
 info structure.
 *********************************************************************
 */
static void
set_order_tracking(E1432_MODULE_LIST_NODE *mn)
{
    struct i1432_chan_info *c;
    int     i, chan, type, order_tracking, active, order_data;

    /* Check all channels to see if any need resampled data */
    order_tracking = 0;
    active = 0;
    order_data = 0;

    for (i = 0; i < 2; i++)
    {
	type = (i == 0) ? E1432_CHAN_TYPE_INPUT : E1432_CHAN_TYPE_TACH;
	c = i1432_chan_list[type];
	for (chan = 0; chan < i1432_chan_count[type]; chan++, c++)
	    if (c->mn == mn)
	    {
		/* Found a channel in this module */
		if (c->active)
		{
		    if (type == E1432_CHAN_TYPE_INPUT)
			active = 1;
		    if (c->calc_data >= E1432_DATA_RESAMP_TIME &&
			c->calc_data <= E1432_DATA_CORR_FREQ)
			order_tracking = 1;
		}
		else
		{
		    if (c->calc_data >= E1432_DATA_RESAMP_TIME &&
			c->calc_data <= E1432_DATA_CORR_FREQ)
			order_data = 1;
		}
	    }
    }

    /* Catch the case where no channels are active */
    if(!active && order_data)
	order_tracking = 1;

    mn->order_tracking = order_tracking;
}

/*
 *********************************************************************
 We cache the value of blocksize, append_status, data_size, and
 meas_type for each module in a measurement, so that
 e1432_read_xxx_data doesn't have to query the module each time.  This
 function fills in the cached values, and is called at the start of a
 measurement.  This is a separate function because we need to EXPORT
 it for unusual applications.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_cached_parm_update(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE **modlist;
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_CHAN_LIST_NODE node;
    SHORTSIZ16 error, cid, data_port, arm_mode, avg_mode, zoom;
    int     mod, modcount, type, count;

    TRACE_PRINTF(0, ("e1432_cached_parm_update(0x%p, %d)\n",
		     hw, ID));

    /* check if valid ID */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    if (ID < 0)
    {
	gn = i1432_get_group_node(hw, ID);
	modcount = gn->modcount;
	modlist = gn->modlist;

	cn = gn->chanlist;
    }
    else
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
	modcount = 1;
	modlist = &mn;

	node.chanID = ID;
	node.next = NULL;
	cn = &node;
    }

    /* Update cached module parameters */
    for (mod = 0; mod < modcount; mod++)
    {
	mn = modlist[mod];

	error = e1432_get_arm_mode(hw, mn->chan_id, &arm_mode);
	if (error)
	    return error;
	mn->rpm_arm = (arm_mode >= E1432_ARM_RPM_RUNUP &&
		       arm_mode <= E1432_ARM_RPM_DELTA);

	error = e1432_get_blocksize(hw, mn->chan_id, &mn->blocksize);
	if (error)
	    return error;
	error = e1432_get_append_status(hw, mn->chan_id,
					&mn->append_status);
	if (error)
	    return error;
	error = e1432_get_data_size(hw, mn->chan_id, &mn->data_size);
	if (error)
	    return error;
	error = e1432_get_meas_type(hw, mn->chan_id, &mn->meas_type);
	if (error)
	    return error;

	error = e1432_get_avg_mode(hw, mn->chan_id, &avg_mode);
	if (error)
	    return error;
	mn->freq_mag_squared = (avg_mode == E1432_AVG_RMS ||
				avg_mode == E1432_AVG_EXPO ||
				avg_mode == E1432_AVG_PEAK);

	error = e1432_get_zoom(hw, mn->chan_id, &zoom);
	if (error)
	    return error;
	mn->zoom = zoom == E1432_ZOOM_ON;

	error = e1432_get_data_port(hw, mn->chan_id, &data_port);
	if (error)
	    return error;

#ifndef	NO_D32
	/* When using local bus, D32 backplane transfers will corrupt
	   the local bus data.  This is not an E1432 problem, it is a
	   generic problem with local bus having to do with the pinout
	   of the VXI P2 connector.  To avoid problems, we switch to
	   D16 transfers if the local bus will be used in this
	   measurement. */
	if (data_port != E1432_SEND_PORT_VME)
	    mn->d32 = 0;
#endif
    }

    /* Update cached channel parameters */
    while (cn != NULL)
    {
	cid = cn->chanID;
	cn = cn->next;

	/* Get the channel type and count */
	type = (cid & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (cid & E1432_CHAN_MASK) - 1;

	/* Do error checking */
	if (cid <= 0 ||
	    type < 0 || type >= E1432_CHAN_TYPES ||
	    count < 0 || count >= i1432_chan_count[type])
	    return i1432_print_error(ERR1432_NO_CHANNEL);

	error = e1432_get_active(hw, cid,
				 &i1432_chan_list[type][count].active);
	if (error)
	    return error;
	error = e1432_get_calc_data(hw, cid,
				    &i1432_chan_list[type][count].calc_data);
	if (error)
	    return error;
	error = e1432_get_scale(hw, cid,
				&i1432_chan_list[type][count].scale);
	if (error)
	    return error;
    }

    /* Go back through the modules and set the order tracking flag.
       This must be done after the above channel loop gets the
       calc_data and active parameters. */
    for (mod = 0; mod < modcount; mod++)
	set_order_tracking(modlist[mod]);

    if (ID < 0)
    {
	/* Set "otra" flag in group, if any module in group is doing
	   order tracking or RPM arming. */
	gn->otra = 0;
	for (mod = 0; mod < modcount; mod++)
	{
	    mn = modlist[mod];
	    if (mn->order_tracking || mn->rpm_arm)
	    {
		gn->otra = 1;
		break;
	    }
	}
    }

    return 0;
}

static SHORTSIZ16
equalize_fifos(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    LONGSIZ32 fifo_size;

    /* If not a group, we're equalized */
    if (ID > 0)
	return 0;

    /* Redundant?  We already did this in init_measure_to_booted */
    gn = i1432_get_group_node(hw, ID);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);

    /* If only one module, we're equalized */
    if (gn->modcount == 1)
	return 0;

    /* This is the maximum fifo size the modules say they can handle,
       given the current setup */
    error = e1432_get_fifo_size_current_max(hw, ID, &fifo_size);
    if (error)
	return error;

    /* Set them all the same */
    error = e1432_set_fifo_size(hw, ID, fifo_size);
    if (error)
	return error;

    return 0;
}

/*
 *********************************************************************
 If any module is in RPM arm mode, then we need soft sync.  Else, if
 any input channel is active and has a calc_data mode that requires
 resampling, then we need soft sync.  The modules will use this only
 if they are trigger slaves, in which case they will not toggle the
 sync/trigger line during the measurement.  This assumes that
 e1432_cached_parm_update has already been done.
 *********************************************************************
 */
static SHORTSIZ16
update_soft_sync(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE **modlist;
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_CHAN_LIST_NODE node;
    SHORTSIZ16 error, cid, s16;
    LONGSIZ32 flags, newflags;
    int     soft_sync, modcount, mod, type, count;

    if (ID < 0)
    {
	gn = i1432_get_group_node(hw, ID);
	modcount = gn->modcount;
	modlist = gn->modlist;

	cn = gn->chanlist;
    }
    else
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
	modcount = 1;
	modlist = &mn;

	node.chanID = ID;
	node.next = NULL;
	cn = &node;
    }

    soft_sync = 0;
    while (cn != NULL)
    {
	cid = cn->chanID;
	cn = cn->next;

	/* Get the channel type and count */
	type = (cid & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (cid & E1432_CHAN_MASK) - 1;

	/* Ignore inactive channels */
	if (i1432_chan_list[type][count].active != E1432_CHANNEL_ON)
	    continue;

	/* Check for resampling modes */
	if (i1432_chan_list[type][count].calc_data >=
	    E1432_DATA_RESAMP_TIME &&
	    i1432_chan_list[type][count].calc_data <=
	    E1432_DATA_CORR_FREQ)
	{
	    soft_sync = 1;
	    break;
	}
    }
    if (!soft_sync)
	for (mod = 0; mod < modcount; mod++)
	{
	    mn = modlist[mod];

	    /* Check for RPM arm modes */
	    error = e1432_get_arm_mode(hw, mn->chan_id, &s16);
	    if (error)
		return error;
	    if (s16 >= E1432_ARM_RPM_RUNUP &&
		s16 <= E1432_ARM_RPM_DELTA)
	    {
		soft_sync = 1;
		break;
	    }
	}

    /* Write new soft_sync value to module */
    for (mod = 0; mod < modcount; mod++)
    {
	mn = modlist[mod];

	error = e1432_read32_register(hw, mn->chan_id,
				      E1432_FLAGBITS_REG, &flags);
	if (error)
	    return error;
	if (soft_sync)
	    newflags = flags | E1432_FLAGBITS_SOFT_SYNC;
	else
	    newflags = flags & ~E1432_FLAGBITS_SOFT_SYNC;
	if (flags != newflags)
	{
	    error = e1432_write32_register(hw, mn->chan_id,
					   E1432_FLAGBITS_REG,
					   newflags);
	    if (error)
		return error;
	}
    }

    return 0;
}

/*
 *********************************************************************
 Scan all channels.  If any have triggering turned on, then the
 trigger source is from that type of channel.  Check the external
 trigger from all modules.  If any have it turned on, then the trigger
 source is external.  Otherwise it's manual.

 Once the trigger source is determined, all modules referred to by the
 ID are informed.

 This assumes that e1432_cached_parm_update has already been done.
 *********************************************************************
 */
static SHORTSIZ16
set_trigger_source(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_CHAN_LIST_NODE node;
    SHORTSIZ16 state, cid;
    int     sources[E1432_CHAN_TYPES] = { 0 };
    int     auto_trigger = 0;
    int     external_trigger = 0;
    int     error, type, count, source;

    if (ID < 0)
    {
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
    }
    else
    {
	node.chanID = ID;
	node.next = NULL;
	cn = &node;
    }

    /* Check each channel to see if it's enabled to trigger */
    while (cn != NULL)
    {
	cid = cn->chanID;
	cn = cn->next;

	/* Get the channel type and count */
	type = (cid & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (cid & E1432_CHAN_MASK) - 1;

	/* Ignore inactive channels */
	if (i1432_chan_list[type][count].active != E1432_CHANNEL_ON)
	    continue;

	/* Check for auto trigger */
	error = e1432_get_auto_trigger(hw, cid, &state);
	if (error)
	    return error;
	if (state == E1432_AUTO_TRIGGER)
	{
	    auto_trigger = 1;
	    /* Auto trigger takes precedence, so quit now */
	    break;
	}

	/* If channel trigger enabled, remember this channel type */
	if (!sources[type])
	{
	    error = e1432_get_trigger_channel(hw, cid, &state);
	    if (error)
		return error;
	    if (state == E1432_CHANNEL_ON)
		sources[type] = 1;
	}

	/* Also check for external trigger */
	if (!external_trigger)
	{
	    error = e1432_get_trigger_ext(hw, cid, &state);
	    if (error)
		return error;
	    if (state != E1432_TRIGGER_EXT_OFF)
		external_trigger = 1;
	}
    }

    /* Auto trigger takes precedence.  If it's not set, use the first
       channel trigger source we come across.  If none are set, check
       for external triggering.  Most of the time, there's only one
       trigger source.  If there's more than one, our trigger delay
       correction will be suspect, but the most likely scenario is
       input+external (to get a sort-of gated trigger) and in this
       case we probably want to go with input as the trigger
       source. */
    if (auto_trigger)
	source = E1432_TRIGGER_SOURCE_AUTO;
    else
    {
	source = E1432_TRIGGER_SOURCE_MANUAL;
	for (type = 0; type < E1432_CHAN_TYPES + 1; type++)
	    if (sources[type])
	    {
		switch(type)
		{
		case E1432_CHAN_TYPE_INPUT:
		    source = E1432_TRIGGER_SOURCE_INPUT;
		    break;
		case E1432_CHAN_TYPE_SOURCE:
		    source = E1432_TRIGGER_SOURCE_SOURCE;
		    break;
		case E1432_CHAN_TYPE_TACH:
		    source = E1432_TRIGGER_SOURCE_TACH;
		    break;
		case E1432_CHAN_TYPE_OTHER:
		    source = E1432_TRIGGER_SOURCE_EXTERNAL;
		    break;
		}
		break;
	    }
	/* If still unset, check external triggering */
	if (source == E1432_TRIGGER_SOURCE_MANUAL && external_trigger)
	    source = E1432_TRIGGER_SOURCE_EXTERNAL;
    }

    return i1432_set_mod(hw, ID, E1432_TRIGGER_SOURCE_REG,
			 E1432_CMD_SET_TRIGGER_SOURCE, source);
}

/*
 *********************************************************************
 After a call is being made to e1432_init_measure(), the system will
 further proceed into the measurement as arm and trigger events occur.
 Returns 0 if all ok, else return negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_init_measure(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;

#if !(defined HPVXI_DOWNLOAD || defined _WINDLL || defined _WIN32)
    extern const char *_e1432buildId1;

    /* make optimizer keep this reference */
    _e1432buildId1++;
#endif

    TRACE_PRINTF(0, ("e1432_init_measure(0x%p, %d)\n", hw, ID));

    /* take boards to booted */
    error = e1432_init_measure_to_booted(hw, ID,1); /* wait for state change */
    if (error)
	return error;

    /* take boards to idle */
    error = e1432_init_measure_finish(hw, ID,1); /* wait for state change */
    if (error)
	return error;

    return 0;
}

/*
 *********************************************************************
 This is done when starting any measurement, before setting up the
 clock and sync/trigger lines in all the modules.
 *********************************************************************
 */
static SHORTSIZ16
before_clock_setup(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;

    /* Reset any running measurements.  Do this before setting up the
       clocks so we don't cause problems in any modules that might
       currently have a measurement running.  Note that we do another
       reset_measure after setting up the clocks. */
    error = e1432_reset_measure(hw, ID);
    if (error)
	return error;

    /* Set the system module for this group */
    return i1432_assign_system_module(hw, ID);
}

/*
 *********************************************************************
 This is done when starting any measurement, after setting up the
 clock and sync/trigger lines in all the modules but before actually
 telling the modules to start the measurement.
 *********************************************************************
 */
static SHORTSIZ16
after_clock_setup(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;

    /* Reset measurement state.  It is necessary to do this after
       setting up the clock and sync lines, so that all modules have a
       sample clock to use for resetting the measurement state. */
    error = e1432_reset_measure(hw, ID);
    if (error)
	return error;

    /* Unfreeze the measurement state machine, so that the modules can
       really start a measurement. */
    error = unreset_measure(hw, ID);
    if (error)
	return error;

    /* Ensure all channels have the same size FIFO */
    error = equalize_fifos(hw, ID);
    if (error)
	return error;

    /* Set up cached parameters */
    error = e1432_cached_parm_update(hw, ID);
    if (error)
	return error;

    /* Tell all modules whether the measurement uses "soft sync".  If
       it does, then all trigger slave modules will not toggle the
       sync/trigger line while going through their measurement loops.
       We rely on passing tach and trigger values between modules to
       keep them in sync.  This assumes that e1432_cached_parm_update
       has already been done. */
    error = update_soft_sync(hw, ID);
    if (error)
	return error;

    /* Tell all modules what the trigger source is.  This assumes that
       e1432_cached_parm_update has already been done. */
    return set_trigger_source(hw, ID);
}

/*
 *********************************************************************
 This is done after telling the modules to start a measurement.  All
 of this could actually be done in after_clock_setup above, but
 putting it here allows us to do some things while we might be waiting
 for the modules anyway.
 *********************************************************************
 */
static SHORTSIZ16
after_start(E1432ID hw, SHORTSIZ16 ID)
{
    i1432_kludge_data_rpm_flag = 0; /* No master trigger yet */
    i1432_kludge_enable_flag = 0; /* Ready for next scan */
    i1432_kludge_tach_flag = 1;	/* Slave tach used, need new trigger */

    /* Must come after e1432_cached_parm_update */
    return map_fifos(hw, ID);
}

/*
 *********************************************************************
 Stuff boards with set up data from the module structures assign one
 module to be the sync driver and drive all modules to the booted
 state.  Returns 0 if all ok, else return negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_init_measure_to_booted(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 wait_after)
{
    E1432_MODULE_LIST_NODE *mn, *sysmn, *trigmn;
    E1432_GROUP_LIST_NODE *gn;
    FLOATSIZ32 freq;
    SHORTSIZ16 error;
    int     mod, modcount;
    struct e1432_tach_parms tach_parms[E1432_TACH_CHANS];

    TRACE_PRINTF(0, ("e1432_init_measure_to_booted(0x%p, %d, %d)\n",
		     hw, ID, wait_after));

    error = before_clock_setup(hw, ID);
    if (error)
	return error;

    /* get system module for this group */
    error = i1432_get_system_module(hw, ID, &sysmn);
    if (error)
	return error;

    /* set up intermodule signals if allowed */
    if (sysmn != NULL && sysmn->autoGroup == E1432_AUTO_GROUP_MEAS_ON)
    {
	/* Count how many modules are used by this group */
	if (ID < 0)
	{
	    gn = i1432_get_group_node(hw, ID);
	    trigmn = gn->trigmn;
	    modcount = gn->modcount;
	}
	else
	{
	    /* Channel ID, so only one module */
	    gn = NULL;
	    trigmn = NULL;
	    modcount = 1;
	}

	TRACE_PRINTF(3, ("init_measure_to_booted, group %d uses %d module\n",
			 ID, modcount));

	if (modcount == 1)
	{
	    /* Set module for internal sync/trigger, and sample0 clock */
	    error = e1432_set_multi_sync(hw, ID,
					 E1432_MULTI_SYNC_OFF);
	    if (error)
		return error;
	    error = e1432_set_clock_master(hw, ID,
					   E1432_MASTER_CLOCK_OFF);
	    if (error)
		return error;
	    error = e1432_set_clock_source(hw, ID,
					   E1432_CLOCK_SOURCE_INTERNAL);
	    if (error)
		return error;
	}
#ifndef HPVXI_DOWNLOAD  /* only 1 module, so save space */
	else if (gn->modcount > 1)
	{
	    /* If module group is master/slave transfer tach parameters */
	    if(trigmn)
	    {
		/* Read tach parms from trigger module */
		error = i1432_xfer_block(trigmn,
					 (unsigned long *) E1432_TACH_PARMS,
					 (unsigned long *) tach_parms,
					 sizeof (tach_parms) / 4, 0, 1);
		if (error)
		    return error;

		/* There is no need to swap the data or check the
		   mn->d32 flag, because we are not keeping the data,
		   we are just transferring it to other modules.  Any
		   swapping that was done when reading the data will
		   get undone during the write. */

	        for (mod = 0; mod < gn->modcount; mod++)
	        {
		    mn = gn->modlist[mod];
		    if(mn == trigmn)
			continue;

		    /* Write tach parms to non-trigger module */
		    error = i1432_xfer_block(mn, (unsigned long *) tach_parms,
					     (unsigned long *)
					     E1432_TACH_PARMS,
					     sizeof (tach_parms) / 4,
					     0, 0);
		    if (error)
			return error;
	        }
	    }

	    /* Set modules for VXI sync/trigger and sample0 clock */
	    error = e1432_set_multi_sync(hw, ID, E1432_MULTI_SYNC_ON);
	    if (error)
		return error;

	    /* Get system module clock freq */
	    error = e1432_get_clock_freq(hw,
					 i1432_get_chan_from_module(sysmn),
					 &freq);
	    if (error)
		return error;

	    /* All non-system-modules don't drive sample0 clock on
	       VXI, and do get their sample0 clock from VXI, and have
	       the same clock frequency as the master. */
	    for (mod = 0; mod < gn->modcount; mod++)
	    {
		mn = gn->modlist[mod];
		if (mn != sysmn)
		{
		    error = e1432_set_clock_master
			(hw, i1432_get_chan_from_module(mn),
			 E1432_MASTER_CLOCK_OFF);
		    if (error)
			return error;
		    error = e1432_set_clock_source
			(hw, i1432_get_chan_from_module(mn),
			 E1432_CLOCK_SOURCE_VXI);
		    if (error)
			return error;
		    error = e1432_set_clock_freq
			(hw, i1432_get_chan_from_module(mn), freq);
		    if (error)
			return error;
		}
	    }

	    /* System module drives sample0 clock on VXI */
	    error = e1432_set_clock_master(hw,
					   i1432_get_chan_from_module(sysmn),
					   E1432_MASTER_CLOCK_ON);
	    if (error)
		return error;
	    error = e1432_set_clock_source(hw,
					   i1432_get_chan_from_module(sysmn),
					   E1432_CLOCK_SOURCE_INTERNAL);
	    if (error)
		return error;
	}
#endif /* HPVXI_DOWNLOAD */
	else
	{
	    return i1432_print_error(ERR1432_NO_CHANNEL);
	}
    }
    else if (sysmn != NULL &&
	     sysmn->autoGroup != E1432_AUTO_GROUP_MEAS_OFF)
	return i1432_la_print_error(sysmn->la,
				    ERR1432_ILLEGAL_GROUP_MEAS_STATE);

    error = after_clock_setup(hw, ID);
    if (error)
	return error;

    /* assert sync */
    error = i1432_sync_system(hw, ID);
    if (error)
	return error;

    error = after_start(hw, ID);
    if (error)
	return error;

    if (wait_after)
    {
        /* Make sure all modules are in BOOTED or greater.  The state
	   could be greater than BOOTED if interrupts are enabled and
	   an interrupt handler calls e1432_init_measure_to_finish
	   before this routine notices that we reached BOOTED. */
        error = i1432_wait_all_greater_state(hw, ID,
					     E1432_MEAS_STATE_BOOTING_WAIT2,
					     DEFAULT_WAIT);
        if (error)
	    return error;
        error = i1432_error_check_group(hw, ID);
        if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 Finish starting the measurment, see init_measure.  Returns 0 if all
 ok, else return negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_init_measure_finish(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 wait_after)
{
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, chan;
    FLOATSIZ32 filter_settling_time, span;
    SHORTSIZ16 zoom;
    FLOATSIZ64 settletime, maxsettletime;
    int     mod, modcount;

    TRACE_PRINTF(0, ("e1432_init_measure_finish(0x%p, %d, %d)\n",
		     hw, ID, wait_after));

    i1432_trigger_sent = 0;	/* Clear flag used in e1432_send_trigger */

    /* make sure all modules are in BOOTED */
    error = i1432_wait_all_state(hw, ID, E1432_MEAS_STATE_BOOTED,
				 DEFAULT_WAIT);
    if (error)
	return error;

    /* clear overload flags */
    error = e1432_check_overloads(hw, ID, 0, 0, 0, 0);
    if (error)
	return error;

    /* assert sync to move modules to idle (or beyond in auto) */
    error = i1432_sync_system(hw, ID);
    if (error)
	return error;

    if (wait_after)
    {
	/* Calculate settling time for each module and keep the longest */
	maxsettletime = E1432_AC_SETTLING_TIME;
	if (ID < 0)
	{
	    gn = i1432_get_group_node(hw, ID);
	    modcount = gn->modcount;
	}
	else
	{
	    gn = NULL;
	    modcount = 1;
	}
	for (mod = 0; mod < modcount; mod++)
	{
	    /* Do one module */
	    if (gn != NULL)
		chan = i1432_get_chan_from_module(gn->modlist[mod]);
	    else
		/* No group, use the channel ID passed in */
		chan = ID;

	    error = e1432_get_span(hw, chan, &span);
	    if (error)
		return error;
	    error = e1432_get_zoom(hw, chan, &zoom);
	    if (error)
		return error;
	    error = e1432_get_filter_settling_time(hw, chan,
						   &filter_settling_time);
	    if (error)
		return error;

	    /* Digital filter settling part...
	       ...plus dma fifo latency, 240 max points (1 chan) */
	    settletime = (FLOATSIZ32)(E1432_DF_SETTLING + 240) / (span * 2.56);
	    if ( zoom == E1432_ZOOM_ON ) /* Account for zoom... */
	    {
		settletime *= 2.0;
	    }
	    settletime += filter_settling_time;	/* user specified settling */
	    settletime += 2;	/* 2 more seconds margin */

	    /* save largest settlingtime */
	    TRACE_PRINTF(3, ("module %d  settletime = %g\n",
			     mod + 1, settletime));
	    if (settletime > maxsettletime)
		maxsettletime = settletime;
	}
	TRACE_PRINTF(5, ("maxsettletime = %g\n", maxsettletime));

	error = i1432_wait_all_greater_state(hw, ID,
					     E1432_MEAS_STATE_SYNC_WAIT1,
					     maxsettletime);
	if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 Turns off any signals the group modules may be driving on the bus.
 Returns 0 if all ok, else return negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_finish_measure(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_finish_measure(0x%p, %d)\n", hw, ID));

    error = e1432_reset_measure(hw, ID);
    if (error)
	return error;

    error = e1432_set_multi_sync(hw, ID, E1432_MULTI_SYNC_OFF);
    if (error)
	return error;

    error = e1432_set_clock_master(hw, ID, E1432_MASTER_CLOCK_OFF);
    if (error)
	return error;

    error = e1432_set_clock_source(hw, ID,
				   E1432_CLOCK_SOURCE_INTERNAL);
    if (error)
	return error;

    error = e1432_set_lbus_mode(hw, ID, E1432_LBUS_MODE_PIPE);
    if (error)
	return error;

    error = e1432_reset_lbus(hw, ID, E1432_RESET_LBUS_OFF);
    if (error)
	return error;

#ifndef HPVXI_DOWNLOAD	/* not using irqs in HPVXI_DOWNLOAD environment */
    error = e1432_set_interrupt_mask(hw, ID, 0);
    if (error)
	return error;

    error = e1432_set_interrupt_priority(hw, ID,
					 E1432_IRQ_PRIORITY_NONE);
    if (error)
	return error;
#endif /* HPVXI_DOWNLOAD */

    return 0;
}

static SHORTSIZ16
calc_pre_arm_wait_single(E1432ID hw, SHORTSIZ16 ID,
		     FLOATSIZ64 *maxsettletime)
{
    double settletime;
    FLOATSIZ32 cfreq;
    LONGSIZ32 decimation;
    SHORTSIZ16 error, data_size;

    /* Do one module */
    error = e1432_get_clock_freq(hw, ID, &cfreq);
    if (error)
	return error;
    error = e1432_get_decimation(hw, ID, &decimation);
    if (error)
	return error;
    error = e1432_get_data_size(hw, ID, &data_size);
    if (error)
	return error;

    /* Calculate time to settling of fifo */
    settletime = 1.5 * MAX_SETTLING * decimation / cfreq;
    if(data_size == E1432_DATA_SIZE_16)
	settletime *= 2.0;

    TRACE_PRINTF(3, ("id %d   pretriggertime = %g\n",
		     ID, settletime));

    /* Save largest settlingtime */
    if (settletime > *maxsettletime)
	*maxsettletime = settletime;

    return 0;
}

static SHORTSIZ16
calc_pre_arm_wait(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ64 *maxsettletime)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, chan;
    int     mod;

    /* Calculate settling time for each module, keep the largest */
    *maxsettletime = MIN_WAIT;

    if (ID < 0)
    {
	/* Do each module in group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    mn = gn->modlist[mod];
	    chan = i1432_get_chan_from_module(mn);

	    error = calc_pre_arm_wait_single(hw, chan, maxsettletime);
	    if (error)
		return error;
	}
    }
    else
    {
	error = calc_pre_arm_wait_single(hw, ID, maxsettletime);
	if (error)
	    return error;
    }

    TRACE_PRINTF(3, ("maxpretrigger = %g\n", *maxsettletime));

    return 0;
}

/*
 *********************************************************************
 Pre-arm a module or group by sending sync.  If the module or group is
 not already in the PRE_ARM state, we wait awhile for PRE_ARM, and
 then error if we don't find it.  Returns 0 if all ok, else negative
 error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_pre_arm_measure(E1432ID hw, SHORTSIZ16 ID,
		      SHORTSIZ16 wait_after)
{
    SHORTSIZ16 error;
    FLOATSIZ64 settletime;

    TRACE_PRINTF(0, ("e1432_pre_arm_measure(0x%p, %d, %d)\n",
		     hw, ID, wait_after));

    error =  calc_pre_arm_wait(hw, ID, &settletime);
    if (error)
	return error;
    
    /* Wait for PRE_ARM state */
    error = i1432_wait_all_greater_state(hw, ID, E1432_MEAS_STATE_SYNC_WAIT1,
				 settletime);
    if (error)
	return error;

    /* Send pre-arm command to modules in group */
    error = i1432_set_mod(hw, ID, -1, E1432_CMD_PRE_ARM_MEASURE, 0);
    if (error)
	return error;

    if (wait_after)
    {
	/* Make sure it left PRE_ARM state */
	error = i1432_wait_all_greater_state(hw, ID,
					     E1432_MEAS_STATE_PRE_ARM_WAIT2,
					     DEFAULT_WAIT);
	if (error)
	    return error;
    }

    return 0;
}

static SHORTSIZ16
calc_arm_wait_single(E1432ID hw, SHORTSIZ16 ID,
		     FLOATSIZ64 *maxsettletime)
{
    double settletime;
    FLOATSIZ32 cfreq;
    LONGSIZ32 tdelay, decimation;
    SHORTSIZ16 error;

    /* Do one module */
    error = e1432_get_clock_freq(hw, ID, &cfreq);
    if (error)
	return error;
    error = e1432_get_decimation(hw, ID, &decimation);
    if (error)
	return error;
    error = e1432_get_trigger_delay(hw, ID, &tdelay);
    if (error)
	return error;

    /* Convert pre-trigger to positive */
    tdelay = -tdelay;
    /* Pretend post-trigger is tiny pre-trigger */
    if (tdelay < 0)
	tdelay = 1;

    /* Calculate time to fill pretrigger fifo */
    settletime = 2 + tdelay * 1.5 * cfreq / decimation;

    TRACE_PRINTF(3, ("id %d   pretriggertime = %g\n",
		     ID, settletime));

    /* Save largest settlingtime */
    if (settletime > *maxsettletime)
	*maxsettletime = settletime;

    return 0;
}

static SHORTSIZ16
calc_arm_wait(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ64 *maxsettletime)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, chan;
    int     mod;

    /* Calculate pretrigger time for each module, keep the largest */
    *maxsettletime = MIN_WAIT;

    if (ID < 0)
    {
	/* Do each module in group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    mn = gn->modlist[mod];
	    chan = i1432_get_chan_from_module(mn);

	    error = calc_arm_wait_single(hw, chan, maxsettletime);
	    if (error)
		return error;
	}
    }
    else
    {
	error = calc_arm_wait_single(hw, ID, maxsettletime);
	if (error)
	    return error;
    }

    TRACE_PRINTF(3, ("maxpretrigger = %g\n", *maxsettletime));

    return 0;
}

static SHORTSIZ16
do_arm_wait(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 state)
{
    FLOATSIZ64 maxsettletime;
    SHORTSIZ16 error;

    /* How long to wait? */
    error = calc_arm_wait(hw, ID, &maxsettletime);
    if (error)
	return error;

    /* Wait for modules to get past <state> */
    error = i1432_wait_all_greater_state(hw, ID, state, maxsettletime);
    if (error)
	return error;

    return 0;
}

static SHORTSIZ16
multimain_master(E1432ID hw, SHORTSIZ16 ID, int freezeval)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;

    if (ID < 0)
    {
	/* Find just one channel in this group */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	if (cn == NULL)
	    return 0;
	ID = cn->chanID;
    }

    /* Prevent one module in this mainframe from progressing past
       the next WAIT1 state, which will hold off this mainframe
       and any which are slaves of it. */
    return i1432_write_cmd1(hw, ID, E1432_CMD_SET_FREEZE, freezeval);
}

/*
 *********************************************************************
 This is used only for a multi-mainframe configuration.  This should
 be called with the ID of a channel or group in the master mainframe.
 The calling procedure is:

 e1432_arm_measure_master_setup(hw, master_id);
 e1432_arm_measure(hw, global_id, 0);
 e1432_arm_measure_slave_finish(hw, slave_id);
 e1432_arm_measure_master_finish(hw, master_id);
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_arm_measure_master_setup(E1432ID hw, SHORTSIZ16 ID)
{
    TRACE_PRINTF(0, ("e1432_arm_measure_master_setup(0x%p, %d)\n",
		     hw, ID));
    return multimain_master(hw, ID, 1);
}

SHORTSIZ16 EXPORT
e1432_arm_measure_master_finish(E1432ID hw, SHORTSIZ16 ID)
{
    TRACE_PRINTF(0, ("e1432_arm_measure_master_finish(0x%p, %d)\n",
		     hw, ID));
    return multimain_master(hw, ID, 0);
}

SHORTSIZ16 EXPORT
e1432_arm_measure_slave_finish(E1432ID hw, SHORTSIZ16 ID)
{
    TRACE_PRINTF(0, ("e1432_arm_measure_slave_finish(0x%p, %d)\n",
		     hw, ID));
    return do_arm_wait(hw, ID, E1432_MEAS_STATE_ARM_WAIT1);
}

/*
 *********************************************************************
 Arm a module or group by sending sync.  If the module or group is not
 already in the IDLE state, we wait awhile for IDLE, and then error if
 we don't find it.  Returns 0 if all ok, else negative error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_arm_measure(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 wait_after)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_arm_measure(0x%p, %d, %d)\n",
		     hw, ID, wait_after));

    /* Wait for IDLE state */
    error = i1432_wait_all_state(hw, ID, E1432_MEAS_STATE_IDLE,
				 DEFAULT_WAIT);
    if (error)
	return error;

    /* Assert sync to move group to arm */
    error = i1432_sync_system(hw, ID);
    if (error)
	return error;

    if (wait_after)
	return do_arm_wait(hw, ID, E1432_MEAS_STATE_ARM_WAIT2);
    else
	return 0;
}

/*
 *********************************************************************
 Trigger a module or group by sending sync.  If the module or group is
 not already in the TRIGGER state, we wait awhile for TRIGGER, and
 then error if we don't find it.  Returns 0 if all ok, else negative
 error number.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_trigger_measure(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 wait_after)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_trigger_measure(0x%p, %d, %d)\n",
		     hw, ID, wait_after));

    /* Wait for TRIGGER state */
    error = i1432_wait_all_state(hw, ID, E1432_MEAS_STATE_TRIGGER,
				 DEFAULT_WAIT);
    if (error)
	return error;

    /* Assert sync to move group to convert */
    error = i1432_sync_system(hw, ID);
    if (error)
	return error;

    if (wait_after)
    {
	/* Make sure it left TRIGGER state */
	error = i1432_wait_all_not_state(hw, ID, E1432_MEAS_STATE_TRIGGER,
					 DEFAULT_WAIT);
	if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************

 The next five functions e1432_init_measure_slave_setup, _slave_middle,
 _slave_finish, _master_setup, and _master_finish are used for making
 measurements for the special case of multiple groups of modules sharing
 common TTL lines for clocking and sync.

 These groups will each run in a separate UNIX (NT?)  process (see [1] if
 you care why).

 The application program (the one calling this E1432 C library)
 must establish interprocess communications to allow these five functions
 to be called in the following order.

 One process controlling one of the module groups must be designated as
 the "Master".  This should be the group that is at the head of the TTL
 trigger line routing.  VXI mainframe extenders currently have a one way
 routing of TTL lines between mainframes so it is important to watch
 directions and which group is driving TTL sync.

 

				 TIME
				    
				   V
          "Master process"         |           "Slave process"
                                   |
	      	                   |
    	                           +--------------------+
				   .			|
				   .	--------------------------------
				   .	|e1432_init_measure_slave_setup|
				   .	--------------------------------
				   .			|
    	        +---------------------------------------+
		|                  .
---------------------------------  .
|e1432_init_measure_master_setup|  .
---------------------------------  .
		|                  .
    	        +---------------------------------------+
				   .			|
			    	   .	---------------------------------
				   .	|e1432_init_measure_slave_middle|
				   .	---------------------------------
				   .			|
    	        +---------------------------------------+
		|                  .
---------------------------------- .
|e1432_init_measure_master_finish| .
---------------------------------- .
		|                  .
    	        +---------------------------------------+
				   .			|
				   .	---------------------------------
				   .	|e1432_init_measure_slave_finish|
				   .	---------------------------------
   


 [1].  Why is this so complex?  Why use separate processes?  The E1432
 library, like the E1431 and E1430 libraries that came before,
 expected ALL modules to be registered at the start with
 e143x_assign_channel_numbers() which sequentially assigned channels to
 cards.  Along comes Longblock E32xx, which can only handle one E1562
 thruput disk per "engine" (group of input modules).  So the original
 thinking was to open multiple engines for multiple mainframe
 measurements and just call e143x_assign_channel_numbers() again to add
 more input cards.  BUT e143x_assign_channel_numbers() cannot be called
 again to add input cards and this was overlooked during the original
 longblock design.  The choices were to A) allow more than one thruput disk
 per engine or B) use separate processes for each engine C) modify the E1432
 library to allow multiple e143x_assign_channel_numbers()


 *********************************************************************
 */



SHORTSIZ16 EXPORT
e1432_init_measure_slave_setup(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_init_measure_slave_setup(0x%p, %d)\n", hw, ID));

    error = before_clock_setup(hw, ID);
    if (error)
	return error;

    /* Set modules for VXI sync/trigger and sample0 clock */
    error = e1432_set_multi_sync(hw, ID, E1432_MULTI_SYNC_ON);
    if (error)
	return error;

    /* All modules don't drive sample0 clock on
       VXI, and do get their sample0 clock from VXI */
    error = e1432_set_clock_master (hw, ID, E1432_MASTER_CLOCK_OFF);
    if (error)
	return error;
    error = e1432_set_clock_source (hw, ID, E1432_CLOCK_SOURCE_VXI);
    if (error)
	return error;

    return after_clock_setup(hw, ID);
}


SHORTSIZ16 EXPORT
e1432_init_measure_slave_middle(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_init_measure_slave_middle(0x%p, %d)\n", hw, ID));

    /* check if valid ID */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    error = after_start(hw, ID);
    if (error)
	return error;

    /* make sure all slave modules are in BOOTED */
    error = i1432_wait_all_state(hw, ID, E1432_MEAS_STATE_BOOTED,
				     DEFAULT_WAIT);
    if (error)
        return error;

    /* clear overload flags */
    error = e1432_check_overloads(hw, ID, 0, 0, 0, 0);
    if (error)
	    return error;

    error = i1432_error_check_group(hw, ID);
    if (error)
        return error;

    return 0;
}


SHORTSIZ16 EXPORT
e1432_init_measure_slave_finish(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *sysmn;
    SHORTSIZ16 chan;
    SHORTSIZ16 decimation_bandwidth;
    FLOATSIZ32 filter_settling_time;
    SHORTSIZ16 error;
    FLOATSIZ32 clock_freq;
    FLOATSIZ64 settletime;

    TRACE_PRINTF(0, ("e1432_init_measure_slave_finish(0x%p, %d)\n", hw, ID));

    i1432_trigger_sent = 0;	/* Clear flag used in e1432_send_trigger */

    /* get system module for this group */
    error = i1432_get_system_module(hw, ID, &sysmn);
    if (error)
	return error;

    if (sysmn == NULL)
	return 0;

    /* calculate settle time of system module */
    chan = i1432_get_chan_from_module(sysmn);
    error = e1432_get_decimation_bandwidth(hw, chan, &decimation_bandwidth);
    if (error)
        return error;
    error = e1432_get_filter_settling_time(hw, chan, &filter_settling_time);
    if (error)
        return error;
    error = e1432_get_clock_freq(hw, chan, &clock_freq);
    if (error)
        return error;

    /* 1.5 * digital filter settling */
    settletime = (1L << decimation_bandwidth) / clock_freq;
    settletime *= E1432_DF_SETTLING * 1.5;

    settletime += filter_settling_time;	/* user specified settling */
    settletime += 2;	/* 2 more seconds margin */
    
    TRACE_PRINTF(3, ("settletime = %g\n", settletime));

    error = i1432_wait_all_greater_state(hw, ID,
					 E1432_MEAS_STATE_SYNC_WAIT1,
					 settletime);
    if (error)
        return error;

    error = i1432_error_check_group(hw, ID);
    if (error)
        return error;

    return 0;
}

SHORTSIZ16 EXPORT
e1432_init_measure_master_setup(E1432ID hw, SHORTSIZ16 ID,
				SHORTSIZ16 wait_after)
{
    E1432_MODULE_LIST_NODE *sysmn;
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_init_measure_master_setup(0x%p, %d,%d)\n",
		       hw, ID, wait_after));

    error = before_clock_setup(hw, ID);
    if (error)
	return error;

    /* get system module for this group */
    error = i1432_get_system_module(hw, ID, &sysmn);
    if (error)
	return error;

    /* Set modules for VXI sync/trigger and sample0 clock */
    error = e1432_set_multi_sync(hw, ID, E1432_MULTI_SYNC_ON);
    if (error)
	return error;

    /* All modules don't drive sample0 clock on
       VXI, and do get their sample0 clock from VXI */
    error = e1432_set_clock_master (hw, ID, E1432_MASTER_CLOCK_OFF);
    if (error)
	return error;
    error = e1432_set_clock_source (hw, ID, E1432_CLOCK_SOURCE_VXI);
    if (error)
	return error;
    /* System module drives sample0 clock on VXI */
    if (sysmn != NULL)
    {
	error = e1432_set_clock_master(hw,
				       i1432_get_chan_from_module(sysmn),
				       E1432_MASTER_CLOCK_ON);
	/* The system module will drive sync for all boards in VXI
	   system */
	if (error)
	    return error;
	error = e1432_set_clock_source(hw,
				       i1432_get_chan_from_module(sysmn),
				       E1432_CLOCK_SOURCE_INTERNAL);
	if (error)
	    return error;
    }

    error = after_clock_setup(hw, ID);
    if (error)
	return error;

    /* assert sync to move the whole VXI extended mainframe system to booted */
    error = i1432_sync_system(hw, ID);
    if (error)
        return error;

    error = after_start(hw, ID);
    if (error)
	return error;

    if (wait_after)
    {
        /* Make sure all modules are in BOOTED or greater.  The state
	   could be greater than BOOTED if interrupts are enabled and
	   an interrupt handler calls e1432_init_measure_to_finish
	   before this routine notices that we reached BOOTED. */
        error = i1432_wait_all_greater_state(hw, ID,
					     E1432_MEAS_STATE_BOOTING_WAIT2,
					     DEFAULT_WAIT);
        if (error)
            return error;

        /* clear overload flags */
        error = e1432_check_overloads(hw, ID, 0, 0, 0, 0);
        if (error)
	        return error;

        error = i1432_error_check_group(hw, ID);
        if (error)
            return error;
    }
    
    return 0;
}


SHORTSIZ16 EXPORT
e1432_init_measure_master_finish(E1432ID hw, SHORTSIZ16 ID,
				 SHORTSIZ16 wait_after)
{
    E1432_MODULE_LIST_NODE *sysmn;
    SHORTSIZ16 chan;
    SHORTSIZ16 decimation_bandwidth;
    FLOATSIZ32 filter_settling_time;
    SHORTSIZ16 error;
    FLOATSIZ32 clock_freq;
    FLOATSIZ64 settletime;

    TRACE_PRINTF(0, ("e1432_init_measure_master_finish(0x%p, %d,%d)\n",
			  hw, ID, wait_after));

    i1432_trigger_sent = 0;	/* Clear flag used in e1432_send_trigger */

    if(!wait_after) /* finish up _to_booted items */
    {
        /* clear overload flags */
        error = e1432_check_overloads(hw, ID, 0, 0, 0, 0);
        if (error)
	        return error;

        error = i1432_error_check_group(hw, ID);
        if (error)
            return error;
    }

    if(wait_after)
    {
        /* get system module for this group */
        error = i1432_get_system_module(hw, ID, &sysmn);
        if (error)
	    return error;

	if (sysmn != NULL)
	{
	    /* calculate settle time of system module */
	    chan = i1432_get_chan_from_module(sysmn);
	    error = e1432_get_decimation_bandwidth(hw, chan,
						   &decimation_bandwidth);
	    if (error)
		return error;
	    error = e1432_get_filter_settling_time(hw, chan,
						   &filter_settling_time);
	    if (error)
		return error;
	    error = e1432_get_clock_freq(hw, chan, &clock_freq);
	    if (error)
		return error;

            /* 1.5 * digital filter settling */
            settletime = (1L << decimation_bandwidth) / clock_freq;
	    settletime *= E1432_DF_SETTLING * 1.5;

            settletime += filter_settling_time;	/* user specified settling */
	    settletime += 2;	/* 2 more seconds margin */
        
	    TRACE_PRINTF(3, ("settletime = %g\n", settletime));
	}
	else
	    settletime = 1;
    }

    /* assert sync to move the whole VXI extended mainframe system to idle */
    error = i1432_sync_system(hw, ID);
    if (error)
        return error;

    if(wait_after)
    {
        error = i1432_wait_all_greater_state(hw, ID,
    					     E1432_MEAS_STATE_SYNC_WAIT1,
					     settletime);
        if (error)
            return error;

        error = i1432_error_check_group(hw, ID);
        if (error)
            return error;
    }

    return 0;
}


SHORTSIZ16
e1432_set_meas_type(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 meas_type)
{
    SHORTSIZ16 error;
    SHORTSIZ16 prev_meas_type;

    TRACE_PRINTF(0, ("e1432_set_meas_type(0x%p, %d, %d)\n", hw, ID, meas_type));

    if ( meas_type != E1432_MEAS_TYPE_OCTAVE
      && meas_type != E1432_MEAS_TYPE_OTHER )  /* replace with others */
    {
	/* change to appropriate error name/string */
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_MEAS);
    }

    error = e1432_get_meas_type(hw, ID, &prev_meas_type);
    if ( error )
    {
        return error;
    }

#ifndef HPVXI_DOWNLOAD		/* no Octave in diagnosics */
    if ( meas_type == E1432_MEAS_TYPE_OCTAVE
      && prev_meas_type != E1432_MEAS_TYPE_OCTAVE )
    {
        error = i1432_octave_download(hw, ID);
	if ( error )
	{
	    return error;
	}
    }
#endif	/* HPVXI_DOWNLOAD */

    error = i1432_set_mod(hw, ID, E1432_MEAS_TYPE_REG, E1432_CMD_SET_MEAS_TYPE,
			  (LONGSIZ32) meas_type);
    if (error)
	return error;

#ifndef HPVXI_DOWNLOAD		/* no Octave in diagnosics */
    error = i1432_octave_configure(hw, ID, meas_type);
    if ( error )
    {
        return error;
    }
#endif	/* HPVXI_DOWNLOAD */

    return 0;
}


SHORTSIZ16
e1432_get_meas_type(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 *meas_type)
{
    TRACE_PRINTF(0, ("e1432_get_meas_type(0x%p, %d, 0x%p)\n", hw, ID,
      meas_type));
    return i1432_get_mod_short(hw, ID, E1432_MEAS_TYPE_REG, meas_type);
}
